package runner_test

import (
	"reflect"
	"testing"

	"github.com/stretchr/testify/require"

	"gitlab.com/gitlab-org/step-runner/pkg/di"
	"gitlab.com/gitlab-org/step-runner/pkg/runner"
	"gitlab.com/gitlab-org/step-runner/proto"
)

func TestStepResourceParser_Parse(t *testing.T) {
	t.Run("parses resource type", func(t *testing.T) {
		tests := []struct {
			name         string
			stepRef      *proto.Step_Reference
			expectedType any
		}{
			{
				name: "git",
				stepRef: &proto.Step_Reference{
					Protocol: proto.StepReferenceProtocol_git,
					Url:      "gitlab.com/user/repo",
					Path:     []string{},
					Filename: "step.yml",
					Version:  "main",
				},
				expectedType: (*runner.GitStepResource)(nil),
			},
			{
				name: "local",
				stepRef: &proto.Step_Reference{
					Protocol: proto.StepReferenceProtocol_local,
					Path:     []string{"path", "to", "step_dir"},
					Filename: "step.yml",
				},
				expectedType: (*runner.FileSystemStepResource)(nil),
			},
			{
				name: "dist",
				stepRef: &proto.Step_Reference{
					Protocol: proto.StepReferenceProtocol_dist,
					Path:     []string{"path", "to", "dist", "step"},
					Filename: "step.yml",
				},
				expectedType: (*runner.DistStepResource)(nil),
			},
			{
				name: "dynamic",
				stepRef: &proto.Step_Reference{
					Protocol: proto.StepReferenceProtocol_dynamic,
					Url:      "${{job.VARIABLE}}",
				},
				expectedType: (*runner.DynamicStepResource)(nil),
			},
		}

		for _, test := range tests {
			t.Run(test.name, func(t *testing.T) {
				parser, err := di.NewContainer().StepResourceParser()
				require.NoError(t, err)

				stepResource, err := parser.Parse("/path/to/step", test.stepRef)
				require.NoError(t, err)

				expectedType := reflect.TypeOf(test.expectedType)
				actualType := reflect.TypeOf(stepResource)
				require.Equal(t, expectedType, actualType)
			})
		}
	})
}

func TestStepResourceParser_ParseLocalStep(t *testing.T) {
	tests := []struct {
		name              string
		parentDir         string
		protoPath         []string
		protoStepPath     any
		expectDescription string
	}{
		{
			name:              "legacy proto field",
			parentDir:         "/parent/dir",
			protoPath:         []string{"path", "to", "step_dir"},
			expectDescription: "/parent/dir/path/to/step_dir/step.yml",
		},
		{
			name:              "proto paths field",
			parentDir:         "/parent/dir",
			protoStepPath:     &proto.Step_Reference_Paths{Paths: &proto.PathParts{Parts: []string{"path", "to", "step_dir"}}},
			expectDescription: "/parent/dir/path/to/step_dir/step.yml",
		},
		{
			name:              "proto expression field",
			protoStepPath:     &proto.Step_Reference_PathExp{PathExp: "${{env.STEP_DIR}}"},
			expectDescription: "${{env.STEP_DIR}}/step.yml",
		},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			resParser, err := di.NewContainer().StepResourceParser()
			require.NoError(t, err)

			stepRef := &proto.Step_Reference{
				Protocol: proto.StepReferenceProtocol_local,
				Path:     test.protoPath,
				Filename: "step.yml",
			}

			if val, ok := test.protoStepPath.(*proto.Step_Reference_Paths); ok {
				stepRef.StepPath = val
			}

			if val, ok := test.protoStepPath.(*proto.Step_Reference_PathExp); ok {
				stepRef.StepPath = val
			}

			stepResource, err := resParser.Parse(test.parentDir, stepRef)
			require.NoError(t, err)

			description := stepResource.(*runner.FileSystemStepResource).Describe()
			require.Equal(t, test.expectDescription, description)
		})
	}
}
