# Copyright 1999-2023 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

# @ECLASS: readme.gentoo-r1.eclass
# @MAINTAINER:
# Pacho Ramos <pacho@gentoo.org>
# @AUTHOR:
# Author: Pacho Ramos <pacho@gentoo.org>
# @SUPPORTED_EAPIS: 6 7 8
# @BLURB: install a doc file shown via elog messages
# @DESCRIPTION:
# An eclass for installing a README.gentoo doc file recording tips
# shown via elog messages. With this eclass, those elog messages will only be
# shown at first package installation and a file for later reviewing will be
# installed under /usr/share/doc/${PF}
#
# You need to call readme.gentoo_create_doc in src_install phase and
# readme.gentoo_print_elog in pkg_postinst

if [[ -z ${_README_GENTOO_ECLASS} ]]; then
_README_GENTOO_ECLASS=1

case ${EAPI} in
	6|7|8) ;;
	*) die "${ECLASS}: EAPI ${EAPI:-0} not supported" ;;
esac

# @ECLASS_VARIABLE: DOC_CONTENTS
# @DEFAULT_UNSET
# @DESCRIPTION:
# The information that is used to create the README.gentoo file.

# @ECLASS_VARIABLE: DISABLE_AUTOFORMATTING
# @DEFAULT_UNSET
# @DESCRIPTION:
# If non-empty, DOC_CONTENTS information will be strictly respected,
# not getting it automatically formatted by fmt. If empty, it will
# rely on fmt for formatting and 'echo -e' options to tweak lines a bit.

# @ECLASS_VARIABLE: FORCE_PRINT_ELOG
# @DEFAULT_UNSET
# @DESCRIPTION:
# If non-empty this variable forces elog messages to be printed.

# @ECLASS_VARIABLE: README_GENTOO_SUFFIX
# @DESCRIPTION:
# If you want to specify a suffix for README.gentoo file please export it.
: "${README_GENTOO_SUFFIX:=""}"

# @FUNCTION: readme.gentoo_create_doc
# @DESCRIPTION:
# Create doc file with ${DOC_CONTENTS} variable (preferred) and, if not set,
# look for "${FILESDIR}/README.gentoo" contents. You can use
# ${FILESDIR}/README.gentoo-${SLOT} also.
# Usually called at src_install phase.
readme.gentoo_create_doc() {
	debug-print-function ${FUNCNAME} "${@}"

	if [[ -n "${DOC_CONTENTS}" ]]; then
		if [[ -n "${DISABLE_AUTOFORMATTING}" ]]; then
			echo "${DOC_CONTENTS}" > "${T}"/README.gentoo || die
		else
			local saved_flags=$-
			set -f				# disable filename expansion in echo arguments
			echo -e ${DOC_CONTENTS} | fold -s -w 70 \
				| sed 's/[[:space:]]*$//' > "${T}"/README.gentoo
			assert
			set +f -${saved_flags}
		fi
	elif [[ -f "${FILESDIR}/README.gentoo-${SLOT%/*}" ]]; then
		cp "${FILESDIR}/README.gentoo-${SLOT%/*}" "${T}"/README.gentoo || die
	elif [[ -f "${FILESDIR}/README.gentoo${README_GENTOO_SUFFIX}" ]]; then
		cp "${FILESDIR}/README.gentoo${README_GENTOO_SUFFIX}" "${T}"/README.gentoo || die
	else
		die "You are not specifying README.gentoo contents!"
	fi

	( # subshell to avoid pollution of calling environment
		docinto .
		dodoc "${T}"/README.gentoo
	) || die
	README_GENTOO_DOC_VALUE=$(< "${T}/README.gentoo")
}

# @FUNCTION: readme.gentoo_print_elog
# @DESCRIPTION:
# Print elog messages with "${T}"/README.gentoo contents. They will be
# shown only when package is installed at first time.
# Usually called at pkg_postinst phase.
#
# If you want to show them always, please set FORCE_PRINT_ELOG to a non empty
# value in your ebuild before this function is called.
# This can be useful when, for example, DOC_CONTENTS is modified, then, you can
# rely on specific REPLACING_VERSIONS handling in your ebuild to print messages
# when people update from versions still providing old message.
readme.gentoo_print_elog() {
	debug-print-function ${FUNCNAME} "${@}"

	if [[ -z "${README_GENTOO_DOC_VALUE}" ]]; then
		die "readme.gentoo_print_elog invoked without matching readme.gentoo_create_doc call!"
	elif ! [[ -n "${REPLACING_VERSIONS}" ]] || [[ -n "${FORCE_PRINT_ELOG}" ]]; then
		echo -e "${README_GENTOO_DOC_VALUE}" | while read -r ELINE; do elog "${ELINE}"; done
		elog ""
		elog "(Note: Above message is only printed the first time package is"
		elog "installed. Please look at ${EPREFIX}/usr/share/doc/${PF}/README.gentoo*"
		elog "for future reference)"
	fi
}

fi
