#
#  equerybts.ui.simple.py  --  User interface for equerybts
#                                + Simple version +
#
#   Copyright (c) 2004 Akinori Hattori, All rights reserved.
#
#   Permission is hereby granted, free of charge, to any person obtaining a
#   copy of this software and associated documentation files (the "Software"),
#   to deal in the Software without restriction, including without limitation
#   the rights to use, copy, modify, merge, publish, distribute, sublicense,
#   and/or sell copies of the Software, and to permit persons to whom
#   the Software is furnished to do so, subject to the following conditions:
#
#   The above copyright notice and this permission notice shall be included in
#   all copies or substantial portions of the Software.
#
#   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
#   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
#   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
#   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
#   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#   OTHER DEALINGS IN THE SOFTWARE.
#
#  $Id: simple.py 78 2004-02-01 15:49:22Z hattya $
#

import errno
import os
import signal
import sys

import equerybts.format as eformat
import equerybts.termctl as etermctl
import equerybts.querybts as equerybts
import equerybts.exceptions as eexcept


__module__ = 'equerybts.ui.simple'


class Prompt:

    def __init__(self, pn, pv, nocolor=0):
        self.termctl  = etermctl.Term()
        self.fancy    = etermctl.FancyText(nocolor)
        self.format   = eformat.Format()
        self.querybts = equerybts.QueryBTS(equerybts.eQueryBTSURLopener(pn, pv))
        self.__mode   = ''
        self.__input  = ''
        self.__view   = 0
        self.__str    = 0
        self.__fname  = ''
        self.__result = ''
        self.__bmax   = 0
        self.__lmax   = 0

    def get_pager(self):
        ret = os.environ['PAGER']

        if not ret:
            raise eexcept.NoPager

        else:
            return ret

    def get_input(self):
        input = self.termctl.getch()

        if ord(input) == 3:
            # ctrl-c
            raise KeyboardInterrupt

        elif ord(input) == 26:
            # ctrl-z
            print
            os.kill(os.getpid(), signal.SIGSTOP)

        elif input.isdigit():
            input_tmp = ''

            if not input_tmp == '\b':
                input_tmp = input
                print '\b' + input_tmp,

            while True:
                tmp = self.termctl.getch()

                if ord(tmp) == 127 or tmp == '\b':
                    # backspace or ctrl-h
                    if len(input_tmp) == 1:
                        if not ord(input_tmp) == 127:
                            # not backspace
                            print '\b' * 2 + ' \b',

                        input_tmp = ''

                    elif input_tmp:
                        input_tmp = input_tmp[:-1]
                        print '\b' * (len(input_tmp) + 2) + input_tmp + ' \b',

                elif ord(tmp) == 3:
                    # ctrl-c
                    raise KeyboardInterrupt

                elif ord(tmp) == 13:
                    # enter
                    self.__input = input_tmp
                    break

                elif not tmp.isdigit():
                    # alphabet or other
                    pass

                else:
                    # digit
                    input_tmp += tmp
                    print '\b' * len(input_tmp) + input_tmp,

                if len(input_tmp) < 1:
                    self.get_input()
                    break

        else:
            self.__input = input

    def with_pager(self, f, mode=0):
        if not mode:
            try:
                p = os.popen(self.get_pager(), 'w')
                p.write(f)
                p.close()

            except IOError, e:
                if e.errno != errno.EPIPE:
                    raise eexcept.UnknownError, e.errno

        else:
            term_h = self.termctl.get_height() - 3
            f_list = f.split('\n')

            if term_h < len(f_list):
                base = max = len(f_list) / term_h
                odd  = len(f_list) % term_h
                current = 0

                while max:
                    current = term_h * (base - max)
                    max -= 1

                    print

                    for bug in f_list[current:current + term_h]:
                        print bug

                    textline = '-- %s/%s %s --' % (current + term_h, len(f_list), mode)

                    print
                    print textline,
                    self.get_input()
                    self.termctl.clear_line(textline)

                    if self.__input == 'q':
                        odd = ''
                        break

                else:
                    current = current + term_h

                if odd:
                    print

                    for bug in f_list[current:current + odd]:
                        print bug

            else:
                print
                print f

    def query_number(self, number):
        result = self.querybts.from_number(number)

        self.__lmax = len(result[2])
        self.__view = 1
        self.__mode = 'view_bug'
        self.format.bug_seed(result[0], result[1], result[2])

        if result[0] == 0:
            raise eexcept.BugNotFound, ('number', number)

    def query_string(self, string):
        result = self.querybts.from_string(string)

        self.__bmax = len(result['bug_id'])
        self.__view = 1
        self.__mode = 'select_bug'
        self.__str  = 1
        self.format.buglist_seed(result)

        if self.__bmax == 0:
            raise eexcept.BugNotFound, ('string', string)

    def main(self):
        while True:
            if self.__mode == 'select_bug':
                self.select_bug()

            elif self.__mode == 'view_bug':
                self.view_bug()

            elif self.__mode == 'select_attachment':
                self.select_attachment()

            elif self.__mode == 'handle_attachment':
                self.handle_attachment()

            elif self.__mode == 'handle_binary_file':
                self.handle_binary_file()

            elif self.__mode == 'save_attachment':
                self.save_attachment()

            else:
                break

    def select_bug(self):
        if self.__view:
            self.with_pager(self.format.output_buglist(), 'bugs')

        # reset
        self.__view = 0

        print 'which bug do you want? [?]: ',
        self.get_input()
        print

        if self.__input == 'h' or self.__input == '?':
            # help
            print
            print '  please select the number'
            print
            print '  q  --  quit             '
            print

        elif self.__input == 'q':
            # quit
            self.termctl.flush()
            sys.exit()

        elif self.__input.isdigit() and int(self.__input) <= self.__bmax:
            # select the bug
            number = self.format.output_bug_number(self.__input)
            self.query_number(number)

        else:
            self.__view = 1

    def view_bug(self):
        if self.__view:
            self.with_pager(self.format.output_bug())

        # reset
        self.__view = 0

        print 'what do you want? [?]: ',
        self.get_input()
        print

        if self.__input == 'h' or self.__input == '?':
            # help
            print
            print '  h, ?  --  show this message'
            print '  l     --  attachment select mode'
            print '  b     --  back to bug select mode'
            print '  q     --  quit                   '
            print
            print '  press any key to show bug again'
            print

        elif self.__input == 'l':
            # link
            self.__view = 1
            self.__mode = 'select_attachment'

        elif self.__input == 'b':
            if self.__str:
                self.__view = 1
                self.__mode = 'select_bug'

            else:
                pass

        elif self.__input == 'q':
            # quit
            self.termctl.flush()
            sys.exit(0)

        else:
            self.__view = 1

    def select_attachment(self):
        if self.__view:
            self.with_pager(self.format.output_links(), 'files')

        # reset
        self.__view = 0

        print 'which link do you want? [?]: ',
        self.get_input()
        print

        if self.__input == 'h' or self.__input == '?':
            # help
            print
            print '  please select the number'
            print
            print '  q  --  quit this mode   '
            print

        elif self.__input == 'q':
            # quit link select mode
            print
            self.__mode = 'view_bug'

        elif self.__input.isdigit() and int(self.__input) <= self.__lmax:
            # select the attachment
            id = self.format.output_attachment_id(self.__input)

            try:
                self.result = self.querybts.from_attachment_id(id)
                self.__view = 1
                self.__mode = 'handle_attachment'

            except IOError, e:
                if e.errno == errno.ENOENT:
                    print self.fancy.red('Error: attachment id %s was not found' % id)
                    print

                else:
                    raise eexcept.UnknownError, e.errno

        else:
            self.__view = 1

    def handle_attachment(self):
        if self.__view:
            print
            print self.format.output_attachment_info(self.result[1], self.result[2], self.result[3])

        self.__view = 0

        print 'view or save? [?]: ',
        self.get_input()
        print

        if self.__input == 'h' or self.__input == '?':
            # help
            print
            print '  v  --  view this attachment'
            print '  s  --  save this attachment'
            print '  q  --  quit this mode     '
            print
            print '  press any key to show attachment info again'
            print

        elif self.__input == 'q':
            # quit attachment mode
            self.__view = 1
            self.__mode = 'select_attachment'

        elif self.__input == 's':
            print
            self.__mode = 'save_attachment'

        elif self.__input == 'v':
            # view
            if self.result[1] == 'application/octet-stream':
                print
                self.__mode = 'handle_binary_file'

            else:
                self.with_pager(self.result[0])
                self.__mode = 'select_attachment'

        else:
            self.__view = 1

    def handle_binary_file(self):
        self.__mode = 'select_attachment'

        print 'this attachment seems binary file, do you really want to view? [y/n]',
        self.get_input()
        print

        if self.__input == 'y':
            # yes
            self.__view = 1
            self.with_pager(self.result[0])

        elif self.__input == 'n':
            # no
            print
            self.__mode = 'handle_attachment'

        else:
            self.__mode = 'handle_binary_file'

    def save_attachment(self):
        self.__view  = 1
        self.__mode  = 'select_attachment'
        self.__fname = '%s' % self.result[2]

        print 'save file to: %s  [y/r/n] ' % self.fancy.d_cyan(self.__fname),
        self.get_input()
        print

        if self.__input == 'y':
            # yes
            self.existence_check()

            try:
                f = open(self.__fname, 'w')
                f.writelines(self.result[0])
                f.close()

            except IOError, e:
                raise eexcept.SaveAttachmentError, (e.args, e.filename)

        elif self.__input == 'r':
            # rename
            print
            self.determine_filename()

        elif self.__input == 'n':
            # quit save attachment mode
            self.__mode = 'handle_attachment'

        else:
            self.__mode = 'save_attachment'

    def existence_check(self):
        if os.path.exists(self.__fname):
            while True:
                print
                print '%s is already exists' % os.path.abspath(self.__fname)
                print 'do you want to replace it? [y/n]',
                self.get_input()
                print

                if self.__input == 'y':
                    break

                elif self.__input == 'n':
                    print
                    self.determine_filename()
                    break

    def determine_filename(self):
        self.__fname = raw_input('save file as: ')

        self.existence_check()

        try:
            f = open(self.__fname, 'w')
            f.writelines(self.result[0])
            f.close()

        except IOError, e:
            raise eexcept.SaveAttachmentError, e.errno
