#
#  equerybts.termctl.py  --  Terminal-related classes for equerybts
#
#   Copyright (c) 2004 Akinori Hattori, All rights reserved.
#
#   Permission is hereby granted, free of charge, to any person obtaining a
#   copy of this software and associated documentation files (the "Software"),
#   to deal in the Software without restriction, including without limitation
#   the rights to use, copy, modify, merge, publish, distribute, sublicense,
#   and/or sell copies of the Software, and to permit persons to whom
#   the Software is furnished to do so, subject to the following conditions:
#
#   The above copyright notice and this permission notice shall be included in
#   all copies or substantial portions of the Software.
#
#   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
#   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
#   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
#   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
#   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#   OTHER DEALINGS IN THE SOFTWARE.
#
#  $Id: termctl.py 73 2004-01-26 11:57:38Z hattya $
#

import fcntl
import sys
import struct
import termios
import tty
import re

import equerybts.textproc as etextproc


class FancyText(etextproc.TextProc):

    def __init__(self, nocolor=0):
        if not nocolor:
            self.decoration = {
                'normal':   '\033[0m',
                'bold':     '\033[1m',
                'italic':   '\033[4m',
                'd_black':  '\033[0;30m',
                'd_red':    '\033[0;31m',
                'd_green':  '\033[0;32m',
                'd_brown':  '\033[0;33m',
                'd_blue':   '\033[0;34m',
                'd_purple': '\033[0;35m',
                'd_cyan':   '\033[0;36m',
                'd_gray':   '\033[0;37m',
                'gray':     '\033[1;30m',
                'red':      '\033[1;31m',
                'green':    '\033[1;32m',
                'yellow':   '\033[1;33m',
                'blue':     '\033[1;34m',
                'purple':   '\033[1;35m',
                'cyan':     '\033[1;36m',
                'white':    '\033[1;37m'
                }

        else:
            self.decoration = {
                'normal':   '\033[0m',
                'bold':     '\033[1m',
                'italic':   '\033[4m',
                'd_black':  '',
                'd_red':    '',
                'd_green':  '',
                'd_brown':  '',
                'd_blue':   '',
                'd_purple': '',
                'd_cyan':   '',
                'd_gray':   '',
                'gray':     '',
                'red':      '',
                'green':    '',
                'yellow':   '',
                'blue':     '',
                'purple':   '',
                'cyan':     '',
                'white':    ''
                }

    def decoration_aa_list(self, color, text):
        return self.decoration[color] + \
               self.join_aa_list(text) + \
               self.decoration['normal']

    def list_color(self, text, f=0):
        ret = ''

        for c in self.decoration.keys():
            if f == 1 and re.match('d_\w', c):
                ret += c + ': ' + self.decoration_aa_list(c, text) + '\n'

            elif f == 2 and not re.match('d_\w', c):
                ret += c + ': ' + self.decoration_aa_list(c, text) + '\n'

            elif not f:
                ret += c + ': ' + self.decoration_aa_list(c, text) + '\n'

        return ret.strip()

    def normal(self, *text):
        return self.decoration_aa_list('normal', text)

    def bold(self, *text):
        return self.decoration_aa_list('bold', text)

    def italic(self, *text):
        return self.decoration_aa_list('italic', text)

    def d_black(self, *text):
        return self.decoration_aa_list('d_black', text)

    def d_red(self, *text):
        return self.decoration_aa_list('d_red', text)

    def d_green(self, *text):
        return self.decoration_aa_list('d_green', text)

    def d_brown(self, *text):
        return self.decoration_aa_list('d_brown', text)

    def d_blue(self, *text):
        return self.decoration_aa_list('d_blue', text)

    def d_purple(self, *text):
        return self.decoration_aa_list('d_purple', text)

    def d_cyan(self, *text):
        return self.decoration_aa_list('d_cyan', text)

    def d_gray(self, *text):
        return self.decoration_aa_list('d_gray', text)

    def gray(self, *text):
        return self.decoration_aa_list('gray', text)

    def red(self, *text):
        return self.decoration_aa_list('red', text)

    def green(self, *text):
        return self.decoration_aa_list('green', text)

    def yellow(self, *text):
        return self.decoration_aa_list('yellow', text)

    def blue(self, *text):
        return self.decoration_aa_list('blue', text)

    def purple(self, *text):
        return self.decoration_aa_list('purple', text)

    def cyan(self, *text):
        return self.decoration_aa_list('cyan', text)

    def white(self, *text):
        return self.decoration_aa_list('white', text)


class Term:

    def __init__(sefl):
        pass

    def getch(self):
        # Reference:
        #   http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/134892
        fd = sys.stdin.fileno()
        old_settings = termios.tcgetattr(fd)
        try:
            tty.setraw(sys.stdin.fileno())
            ch = sys.stdin.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)
        return ch

    def get_size(self):
        # Reference:
        #   http://aspn.activestate.com/ASPN/Mail/Message/python-list/530228
        #   http://twistedmatrix.com/pipermail/twisted-commits/2002-November/004771.html
        height, width = struct.unpack('4H', fcntl.ioctl(0, termios.TIOCGWINSZ, '\000'*8))[0:2]
        return height, width

    def get_height(self):
        return self.get_size()[0]

    def get_width(self):
        return self.get_size()[1]

    def clear_line(self, line):
        backward  = len(line) + 2
        cleartext = len(line) + 2

        clear  = '\b' * backward
        clear += ' '  * cleartext
        clear += '\b' * backward
        clear += '\n'

        return sys.stdout.write(clear)

    def flush(self):
        return sys.stdout.write('\033[H')
