#
#  equerybts.format.simple.py  --  Text format class for equerybts
#                                    + Simple version +
#
#   Copyright (c) 2004 Akinori Hattori, All rights reserved.
#
#   Permission is hereby granted, free of charge, to any person obtaining a
#   copy of this software and associated documentation files (the "Software"),
#   to deal in the Software without restriction, including without limitation
#   the rights to use, copy, modify, merge, publish, distribute, sublicense,
#   and/or sell copies of the Software, and to permit persons to whom
#   the Software is furnished to do so, subject to the following conditions:
#
#   The above copyright notice and this permission notice shall be included in
#   all copies or substantial portions of the Software.
#
#   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
#   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
#   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
#   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
#   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#   OTHER DEALINGS IN THE SOFTWARE.
#
#  $Id: simple.py 78 2004-02-01 15:49:22Z hattya $
#

import re

import equerybts.termctl as etermctl
import equerybts.textproc as etextproc
import equerybts.exceptions as eexcept


__module__ = "equerybts.format.simple"


class Format(etextproc.TextProc):

    def __init__(self, nocolor=0):
        self.fields   = ''
        self.comments = ''
        self.links    = ''
        self.buglist  = ''
        self.fancy    = etermctl.FancyText(nocolor)
        self.termctl  = etermctl.Term()
        self.savedata = ''
        self.__width  = int(self.termctl.get_width() * 0.9)
        self.__indent = int((self.termctl.get_width() - self.__width) / 4)

    def save_data(self, *data):
        for d in data:
            self.savedata += d + '\n'

    def free_data(self):
        self.savedata = ''

    def get_data(self):
        return self.savedata

    def bug_seed(self, fields, comments, links):
        self.fields   = fields
        self.comments = comments
        self.links    = links

    def buglist_seed(self, buglist):
        self.buglist = buglist

    def replace_link(self, text, tag=0):
        if tag:
            a_re = re.compile('(#A(\d+)#)([^#]*)(#/A\d+#)')

            for r in a_re.findall(text):
                link = r[0] + '[^#]*' + r[3]
                text = re.sub(link, self.fancy.d_blue('(' + r[1] + ')' + r[2]), text)

        else:
            text = re.sub(text, self.fancy.d_blue(text), text)

        return text

    def format_line(self, indent, *source):
        # flag 1: comment
        # flag 2: summary
        # flag 3: abbr
        flag = fval = url = 0
        ret = ' ' * indent

        for s in source:
            color, text = s

            if color in ['wrap', 'abbr']:
                flag, fval = text
                continue

            elif flag > 0 and flag < 3:
                text = self.wrap_text(self.__width, fval, text)

                if flag == 2:
                    # clean up (summary)
                    text = text.lstrip()

            elif flag == 3:
                text = self.abbreviate_text(fval, text)

            elif text.strip().lower() == 'url:':
                url = 1

            elif url > 0:
                if text:
                    text = self.replace_link(text, tag=0)


            if not color:
                try:
                    color = getattr(self.fancy, 'normal')

                except AttributeError:
                    raise eexcept.UnknownError, ('', 'Unknown error in %s.' % __module__)

            else:
                try:
                    color = getattr(self.fancy, color)

                except AttributeError:
                    raise eexcept.UnknownError, ('', '%s is not defined in %s.' % (color, __module__))

            text = self.replace_link(text, tag=1)

            if flag == 2:
                # summary
                for x in text.strip().split( '\n'):
                    ret += color(x) +'\n'

            else:
                ret += color(text)

        return ret

    def format_buglist(self):
        length = len(self.buglist['bug_id'])
        vr_len = self.__indent * 6 + len(str(length + 1)) + \
                 len(self.buglist['bug_id'][length - 1]) + 5 # + 11
##         ag_length = 0

##         for mail in self.buglist['assigned_to']:
##             if ag_length < len(mail):
##                 ag_length = len(mail)

        for n in range(1, length + 1):
            num = n - 1

            sp_indent = ' ' * self.__indent
            nu_indent = ' ' * (len(str(length)) - len(str(n)))
            id_indent = ' ' * (len(self.buglist['bug_id'][length - 1]) - \
                               len(self.buglist['bug_id'][num]))
##             ag_indent = ' ' * (self.__indent + ag_length - len(self.buglist['assigned_to'][num]))

            status = severity = ''

            # severity
            if self.buglist['bug_severity'][num] == 'critical':
                severity = 'd_red'

            elif self.buglist['bug_severity'][num] == 'blocker':
                severity = 'red'

            elif self.buglist['bug_severity'][num] == 'enancement':
                severity = 'italic'


            # status
            if self.buglist['bug_status'][num].lower() == 'new':
                status = self.buglist['bug_status'][num]

            elif self.buglist['bug_status'][num].lower() in ['resolved', 'verified']:
                status = self.buglist['resolution'][num]

##                 if not status.lower() in ['test-request', 'later']:
##                     severity = 'd_black'

            else:
                status = self.buglist['bug_status'][num]

##                 if status.lower() == 'closed':
##                     severity = 'd_black'

            if len(status) > 4:
                status = status[:4]

            else:
                status = status.ljust(4)


            self.save_data(
                self.format_line(self.__indent,
                                 ('', nu_indent),
                                 ('',  '%s)  ' % n),
                                 ('', id_indent),
                                 ('d_blue', self.buglist['bug_id'][num]),
                                 ('', sp_indent),
                                 (severity, self.buglist['bug_severity'][num][:3]),
                                 ('', sp_indent),
                                 (severity, self.buglist['rep_platform'][num][:3]),
                                 ('', sp_indent),
##                                  (severity, self.buglist['assigned_to'][num]),
##                                  ('', ag_indent),
                                 (severity, status),
                                 ('', sp_indent),
                                 ('abbr', (3, self.__width - vr_len)),
                                 (severity, self.buglist['short_short_desc'][num])
                                 )
                )

    def format_fields(self):
        r_length = len(self.fields['reported-by']) + 10 + 10
        r_indent = self.__width + self.__indent - r_length
        o_indent = r_indent + 2
        m_indent = o_indent - 7
        d_indent = 0

        c_status = status = severity = modified = ''

        opened = self.fields['opened']['date'] + ' ' + \
                 self.fields['opened']['time'] + ' ' + \
                 self.fields['opened']['tz']

        # last modified
        if self.comments.keys():
            k = 'com#' + str(len(self.comments.keys()))
            modified = self.comments[k]['date'] + ' ' + \
                       self.comments[k]['time'] + ' ' + \
                       self.comments[k]['tz']

        else:
            modified = opened

        # status
        if self.fields['status'].lower() == 'new':
            c_status = 'd_green'
            status = self.fields['status']

        elif self.fields['status'].lower() == 'resolved':
            status = self.fields['resolution']

        else:
            status = self.fields['status']

        # severity
        if self.fields['severity'].lower() == 'critical':
            severity = 'd_red'

        elif self.fields['severity'].lower() == 'blocker':
            severity = 'red'

        elif self.fields['severity'].lower() == 'enhancement':
            severity = 'italic'


        self.save_data(
            self.format_line(self.__indent,
                             ('', '-' * self.__width)),
            '',
            self.format_line(self.__indent,
                             ('bold', 'Bug ' + self.fields['bug#']),
                             ('', '  --  '),
                             ('wrap', (2, self.__indent + 10 + len(self.fields['bug#']))),
                             ('bold', self.fields['summary'])),
            '',
            self.format_line(r_indent,
                             ('bold', 'Reporter: '),
                             ('', self.fields['reported-by'])),
            self.format_line(o_indent,
                             ('bold', 'Opened: '),
                             ('', opened)),
            self.format_line(m_indent,
                             ('bold', 'Last Modified: '),
                             ('', modified)),
            '',
            self.format_line(self.__indent,
                             ('bold', '     Status: '),
                             (c_status, status)),
            self.format_line(self.__indent,
                             ('bold', '  Serverity: '),
                             (severity, self.fields['severity'])),
            self.format_line(self.__indent,
                             ('bold', '  Component: '),
                             ('', self.fields['component'])),
            self.format_line(self.__indent,
                             ('bold', '        URL: '),
                             ('', self.fields['url'])),
            self.format_line(self.__indent,
                             ('bold', 'Assigned To: '),
                             ('d_cyan', self.fields['assigned-to'])),
            '',
            self.format_line(self.__indent,
                             ('bold', 'Description: ')),
            '',
            self.format_line(d_indent,
                             ('wrap', (1, self.__indent + 2)),
                             ('', self.fields['description']))
            )

    def format_comments(self):
        c_indent = self.__indent + 5

        for k in range(1, len(self.comments.keys()) + 1):
            k = 'com#' + str(k)
            comh = self.comments[k]
            date = comh['date'] + ' ' + \
                   comh['time'] + ' ' + \
                   comh['tz']

            f_indent = self.__indent + 3


            self.save_data(
                self.format_line(self.__indent,
                                 ('bold', '-' * self.__width)),
                self.format_line(f_indent,
                                 ('bold', 'From: '),
                                 ('', comh['mail']),
                                 ('bold', '   Date: '),
                                 ('', date)),
                '',
                self.format_line(0,
                                 ('wrap', (1, c_indent)),
                                 ('', comh['com']))
                )

        self.save_data(
            self.format_line(self.__indent,
                             ('bold', '-' * self.__width)),
            )

    def format_links(self):

        if len(self.links.keys()) < 1:
            self.save_data(
                self.format_line(self.__indent,
                                 ('', 'nothing.'))
                )

        else:
            length    = len(self.links.keys())

            for n in range(1, length + 1):
                k = 'A' + str(n)

                id_indent = len(self.links['A%s' % str(length)][1]) + 3 - len(self.links[k][1])
                id_indent = ' ' * id_indent

                self.save_data(
                    self.format_line(self.__indent,
                                     ('', str(n) + ')'),
                                     ('d_blue', '  id='),
                                     ('d_blue', self.links[k][1]),
                                     ('', id_indent),
                                     ('d_blue', self.links[k][0]))
                    )

    def format_attachment_info(self, mime, name, size):
        self.save_data(
            self.format_line(self.__indent,
                             ('d_green', 'type: '),
                             ('', mime)),
            self.format_line(self.__indent,
                             ('d_green', 'name: '),
                             ('', name)),
            self.format_line(self.__indent,
                             ('d_green', 'size: '),
                             ('', '%s byte' % size))
            )

    def output_buglist(self):
        self.free_data()
        self.format_buglist()

        return self.get_data()

    def output_bug_number(self, number):
        return self.buglist['bug_id'][int(number) - 1]

    def output_bug(self):
        self.free_data()
        self.format_fields()
        self.format_comments()

        return self.get_data()

    def output_links(self):
        self.free_data()
        self.format_links()

        return self.get_data()

    def output_attachment_id(self, number):
        key = 'A%s' % number

        return self.links[key][1]

    def output_attachment_info(self, mime, name, size):
        self.free_data()
        self.format_attachment_info(mime, name, size)

        return self.get_data()
