#' Deploy a Document
#'
#' Deploys an application consisting of a single R Markdown document or other
#' single file (such as an HTML or PDF document).
#'
#' @param doc Path to the document to deploy.
#' @param ... Additional arguments to \code{\link{deployApp}}. Do not supply
#'   \code{appDir}, \code{appFiles}, or \code{appPrimaryDoc};
#'   these three parameters are automatically generated by \code{deployDoc} from
#'   the document.
#'
#' @details When deploying an R Markdown document, any files which are
#'   required to render and display the file must be deployed.
#'
#'   This method discovers these additional files using
#'   \code{\link[rmarkdown:find_external_resources]{find_external_resources}}
#'   from \pkg{rmarkdown}.
#'
#'   If you find that the document is missing dependencies, either specify the
#'   dependencies explicitly in the document (the documentation for
#'   \code{\link[rmarkdown:find_external_resources]{find_external_resources}}
#'   explains how to do this), or call \code{\link{deployApp}} directly and
#'   specify your own file list in the \code{appFiles} parameter.
#'
#' @export
deployDoc <- function(doc, ...) {
  # validate inputs
  if (!file.exists(doc)) {
    stop("The document '", doc, "' does not exist.")
  }
  if (!require("rmarkdown") ||
      packageVersion("rmarkdown") < "0.5.2") {
    stop("Version 0.5.2 or later of the rmarkdown package is required to ",
         "deploy individual R Markdown documents.")
  }

  # default to deploying just the single file specified
  qualified_doc <- normalizePath(doc, winslash = "/")
  app_files <- basename(qualified_doc)

  # if this document's type supports automated resource discovery, do that now,
  # and add the discovered files to the deployment list
  ext <- tolower(tools::file_ext(doc))
  if (ext %in% c("rmd", "html", "htm")) {
    message("Discovering document dependencies... ", appendLF = FALSE)
    res <- rmarkdown::find_external_resources(qualified_doc)
    message("OK")
    app_files <- c(app_files, res$path)
  }

  # deploy the document with the discovered dependencies
  deployApp(appDir = dirname(qualified_doc),
            appFiles = app_files,
            appPrimaryDoc = basename(qualified_doc),
            ...)
}

